#include "cupsparserview.h"
#include "detailwidget.h"

#include <qpopupmenu.h>
#include <qheader.h>
#include <qmessagebox.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdialogbase.h>

CupsParserView::CupsParserView( QWidget *parent , const char *name )
	: QListView( parent, name )
{
	m_parser = 0;
	addColumn( i18n( "Date" ) );
	setColumnWidth( 0, fontMetrics().width( "2002-08-22 15:09:00MM" ) + 20 );
	setAllColumnsShowFocus( true );
	setShowSortIndicator( true );
	m_limit = 100;
	m_items.setAutoDelete( false );
	m_groupitems.setAutoDelete( true );
	connect( this, SIGNAL( doubleClicked( QListViewItem* ) ), SLOT( slotProperty() ) );
	m_grouping = -1;
	m_restored = false;
}

CupsParserView::~CupsParserView()
{
	delete m_parser;
}

void CupsParserView::setParser( CupsParser *parser )
{
	delete m_parser;
	m_parser = parser;
	if ( m_parser )
	{
		m_items.clear();
		m_groupitems.clear();
		clear();
		while ( columns() > 1 )
			removeColumn( columns()-1 );
		QStringList fields = m_parser->fields();
		for ( QStringList::ConstIterator it=fields.begin(); it!=fields.end(); ++it )
			addColumn( *it );
		connect( m_parser, SIGNAL( newItems( const CupsParser::ItemList& ) ), SLOT( slotNewItems( const CupsParser::ItemList& ) ) );
		connect( m_parser, SIGNAL( restarted() ), SLOT( slotRestarted() ) );
		m_parser->startParsing();
	}
}

QListViewItem* CupsParserView::findGroupItem( const QString& key )
{
	return m_groupitems.find( key );
}

CupsParserViewItem* CupsParserView::createItem( CupsParser::Item *item )
{
	CupsParserViewItem *viewItem = 0;
	if ( m_grouping == -1 )
		viewItem = new CupsParserViewItem( this, item );
	else
	{
		QString grpKey = ( m_grouping == 0 ? item->date.date().toString( "yyyy-MM-dd" ) : item->fields[ m_grouping-1 ].toString() );
		QListViewItem *grpItem = findGroupItem( grpKey );
		if ( !grpItem )
		{
			grpItem = new QListViewItem( this );
			m_groupitems.insert( grpKey, grpItem );
			grpItem->setText( 0, i18n( "Group" ) + " [" + grpKey + "]" );
			if ( m_grouping > 0 )
				grpItem->setText( m_grouping, grpKey );
			grpItem->setPixmap( 0, SmallIcon( "folder" ) );
		}
		viewItem = new CupsParserViewItem( grpItem, item );
	}
	return viewItem;
}

void CupsParserView::slotNewItems( const CupsParser::ItemList& list )
{
	unsigned int nbrOfItems = m_items.count();
	CupsParser::ItemListIterator it( list );

	for ( ; it.current(); ++it )
	{
		CupsParserViewItem *item = createItem( it.current() );
		if ( m_limit >= 0 )
		{
			while ( ( int )m_items.count() >= m_limit )
				delete m_items.take( 0 );
		}
		m_items.append( item );
	}

	if ( m_grouping != -1 )
	{
		QDictIterator<QListViewItem> it( m_groupitems );
		for ( ; it.current(); )
			if ( it.current()->childCount() == 0 )
				m_groupitems.remove( it.currentKey() );
			else
				++it;
	}

	if ( m_items.count() != nbrOfItems )
		emit itemsChanged( ( int )m_items.count() );
}

void CupsParserView::slotRestarted()
{
	m_items.clear();
	m_groupitems.clear();
	clear();
	emit selectionChanged();
}

void CupsParserView::slotGrouping( int ID )
{
	// Do nothing if grouping has not changed
	if ( ID-1 == m_grouping )
		return;

	m_grouping = ID-1;

	if ( m_grouping == -1 )
	{
		// No grouping
		QPtrListIterator<CupsParserViewItem> it( m_items );
		for ( ; it.current(); ++it )
			insertItem( it.current() );
		m_groupitems.clear();
	}
	else
	{
		// Grouping by column m_grouping
		QPtrList<CupsParserViewItem> tmpList = m_items;
		QDict<QListViewItem> newGrpItems;
		newGrpItems.setAutoDelete( false );
		tmpList.setAutoDelete( false );
		while ( tmpList.count() > 0 )
		{
			CupsParserViewItem *item = tmpList.take( 0 );
			QString grpKey = item->groupKey( m_grouping );
			QListViewItem *grpItem = new QListViewItem( this );
			newGrpItems.insert( grpKey, grpItem );
			grpItem->setText( 0, i18n( "Group" ) + " [" + grpKey + "]" );
			if ( m_grouping > 0 )
				grpItem->setText( m_grouping, grpKey );
			grpItem->setPixmap( 0, SmallIcon( "folder" ) );
			grpItem->insertItem( item );
			QPtrListIterator<CupsParserViewItem> it( tmpList );
			for ( int i = 0; it.current(); )
				if ( it.current()->match( m_grouping, grpKey ) )
				{
					item = tmpList.take( i );
					grpItem->insertItem( item );
				}
				else
					++it, ++i;
		}
		m_groupitems = newGrpItems;
	}

	setRootIsDecorated( m_grouping != -1 );
}

int CupsParserView::grouping() const
{
	return m_grouping+1;
}

bool CupsParserView::restored() const
{
	return m_restored;
}

void CupsParserView::setRestored( bool b )
{
	m_restored = b;
}

static QString tableRow( const QString& s1, const QString& s2 )
{
	return ( "<tr><td align=right><b>" + s1 + ":</b></td><td>" + s2 + "</td></tr>" );
}

QString CupsParserView::formatItem( CupsParser::Item *item )
{
	QString s;
	for ( int i=item->fields.size()-1; i>=0; i-- )
		s.prepend( tableRow( columnText( i+1 ), item->fields[ i ].toString() ) );
	s.prepend( tableRow( columnText( 0 ), item->date.toString( Qt::TextDate ) ) );
	s.prepend( "<table>" ).append( "</table>" );
	return s;
}

void CupsParserView::slotProperty()
{
	CupsParserViewItem *item = static_cast<CupsParserViewItem*>( selectedItem() );
	if ( item && item->childCount() == 0 && item->item() )
	{
		CupsParser::Item *pitem = item->item();
		KDialogBase dlg( this, "DetailDialog", true, i18n( "Event Properties" ), KDialogBase::Ok );
		DetailWidget *w = new DetailWidget( formatItem( pitem ), pitem->fulldata, pitem->pixmap, &dlg );
		dlg.setMainWidget( w );
		dlg.exec();
	}
}

int CupsParserView::limit() const
{
	return m_limit;
}

void CupsParserView::setLimit( int val )
{
	m_limit = val;
	/* Should update the view if parsing has started */
}

CupsParser* CupsParserView::parser() const
{
	return m_parser;
}

CupsParserViewItem::CupsParserViewItem( QListView *lv, CupsParser::Item *item )
	: QListViewItem( lv ), m_item( item )
{
	init();
}

CupsParserViewItem::CupsParserViewItem( QListViewItem *parent, CupsParser::Item *item )
	: QListViewItem( parent ), m_item( item )
{
	init();
}

CupsParserViewItem::~CupsParserViewItem()
{
	delete m_item;
}

void CupsParserViewItem::init()
{
	CupsParser::Item *pitem = item();
	if ( pitem )
	{
		setText( 0, pitem->date.toString( "yyyy-MM-dd hh:mm:ss" ) );
		for ( int i = pitem->fields.size(); i >= 1; i-- )
			setText( i, pitem->fields[ i-1 ].toString() );
		if ( !pitem->pixmap.isEmpty() )
			setPixmap( 0, SmallIcon( pitem->pixmap ) );
	}
}

int CupsParserViewItem::compare( QListViewItem *i, int col, bool ascending ) const
{
	CupsParserViewItem *other = static_cast<CupsParserViewItem*>( i );
	switch( col )
	{
		case 0:
			return ( m_item->date < other->m_item->date ? -1 : ( m_item->date > other->m_item->date ? 1 : 0 ) );
		default:
			if ( m_item->fields[ col-1 ].type() == QVariant::String )
				return ( m_item->fields[ col-1 ].asString().compare( other->m_item->fields[ col-1 ].asString() ) );
			else
			{
				int d1 = m_item->fields[ col-1 ].asInt(), d2 = other->m_item->fields[ col-1 ].asInt();
				return ( d1 < d2 ? -1 : ( d1 > d2 ? 1 : 0 ) );
			}
	}
}

QString CupsParserViewItem::groupKey( int col )
{
	if ( col == 0 )
		return m_item->date.date().toString( "yyyy-MM-dd" );
	else
		return text( col );
}

bool CupsParserViewItem::match( int col, const QString& key )
{
	if ( col == 0 )
		return text( 0 ).startsWith( key );
	else
		return text( col ) == key;
}

CupsParser::Item* CupsParserViewItem::item() const
{
	return m_item;
}

#include "cupsparserview.moc"
