#ifndef THREADS_H
#define THREADS_H

// $Revision: 1.7 $

#include <qobject.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

/** Class that provides mutex locks for data.
  *
  * Mutex lock means "mutual access exclusion lock". An object of
  * this class provides thread-safe management of an object of class
  * type. It is templatized to be unique and usable for 
  * (nearly) all kind of C++ representable data types. It was 
  * written while having some glasses of 14 years old Oban whisky
  * what may excuse some design flaws.
  * After creating an object the data is not locked. 
  * "A mutex  is a MUTual EXclusion device, and is useful for
  * protecting shared data structures from concurrent  modifications,
  * and implementing critical sections and monitors."
  * (pthread_mutex_lock man page included in the linuxthreads 
  * package)
  */

template <class type>
class Lock 
{
public:
  Lock(pthread_mutexattr_t* attr=0);
  ~Lock();
  /** Call @ref get to get the value of the data.
    *
    * @ref get tries to get the lock for the data object 
    * (to lock the variable the template class  @ref Lock is instantiated for),
    * then delivers the data back and releases the lock.
    * If the data is not locked it thus returns immidiately.
    * If it is locked @ref get blocks the thread until the previously
    * given lock is released.
    * 
    * If two parts of your software need to access the data they both 
    * call @ref get and will both be allowed to read.
    * The data can be set using the @ref set -method which 
    * naturally also serializes via the lock.
    * Any order is not guaranteed because the underlying 
    * linuxthreads package does not guarantee it. You cannot assume one
    * of your threads to finish earlier than another, anyway, since both run
    * concurrently on the processor.
    */
  type get() const;
  /** @ref set sets the value of the protected data.
    *
    * @param data Call @ref set with a reference to the new value of 
    * data to set a new value. This locks the data, changes it 
    * and unlocks. As long as it is written no other thread can read 
    * it. Because of this a lock should be freed as soon as possible,
    * and that is what @ref set does. If the data needs to be 
    * locked for a longer time use @ref lock and later
    * @ref unlock . Do not forget to unlock a locked variable, this might
    * block your whole executable.
    */
  void set(const type& data);
  /** @ref lock explicitely locks the data. See the documentation for
    * @ref tryLock for explanation.
    */
  type* lock() const;
  /** @ref tryLock tries to get the lock.
    * 
    * @ref tryLock and @ref lock both try to get the lock and 
    * return zero if it is not available.
    * If it is available, the functions return a pointer that  
    * points to the data that is 
    * protected by the lock. This is of course a risk because you can 
    * break the lock by saving the pointer. Because of that you should 
    * use this pointer only as long as you need it, then unlock the 
    * data and delete the pointer in any way. The intention is that 
    * you do not have to call @ref get if you, for example, 
    * protect a large object or an array with the lock, or you need to
    * lock data for more than a singular operation on it. 
    *
    * The difference between @ref tryLock and @ref lock is that 
    * @ref lock blocks the thread if the lock is not available
    * and returns after it has been freed. That is, it only returns zero
    * if an error occured. The only errors that may happen in this 
    * context is that the mutex variable has not been initialized 
    * which is a bug because the constructor has to do that or that the 
    * data object is zero what is of course also a bug.
    *
    * @ref tryLock returns zero if the lock is held by another thread,
    * thus it does not block the calling thread.
    */ 
  type* tryLock() const;
  /** @ref unlock frees the lock. The behaviour of @ref unlock heavily 
    * depends on the POSIX threads package that is used at compilation time
    * because there are packages that do allow unlocking a mutex that is 
    * not locked and others that do not. Linuxthreads does. The best
    * way to avoid such problems is to call @ref unlock only in 
    * conjunction with @ref lock or a succesful @ref tryLock .
    * If you do so, the 
    * number of locks is the same as the number of unlocks except that 
    * your program already crashed before :-)
    */
  void unlock() const;
  /** Operators (inline):
    * Operator @ref "operator type" (a type conversion operator)
    * calls @ref get to get the data, 
    * thus behaves like if you call @ref get directly.
    */
  operator type() const { return get(); }
  /** Operator @ref "operator=" calls @ref set and thus 
    * behaves exactly like it.
    */
  void operator=(const type& temp) { set(temp); }
protected:
  /// data contains the protected data.
  mutable type data;
  /// mutex handles the protection.
  mutable pthread_mutex_t mutex;
};

/** A basic encapsulation for threads. Shortly:
    - derive it, redefining threadMain()
    - put all thread-specific data that is NOT
      shared with other threads in the protected
      object space
    - put all data shared with other threads in
      the protected object space and protect it
      using a lock, get()-methods would be useable
      to serialize access to it (but do not name
      it get...(), or Kalle will beat you up).
    - To handle parameters to the threads main
      function and get the results back, use
      member variables of your thread object. This
      is more safe than converting to void*, anyway.
*/

class Thread 
{
public:
  Thread();
  virtual ~Thread() {}
  /** Creates a thread.
    * Creating an object of a class derived from @ref Thread does 
    * not already start a thread. The programmer has to call 
    * @ref start for this. @ref start does the following things:
    * It starts the thread by calling @ref pthread_create
    * with the appropriate arguments.
    * It stores the thread id of the created thread so that it
    * can be retrieved with the @ref identity function.
    * @return It returns zero if the thread could be started and the value 
    * of @ref errno if an error occured. @ref errno may then 
    * contain two different values defined in errno.h according to the
    * pthread_create manpage.
    * @see pthread_create manpage
    */
  virtual int start();
  /** @ref join waits for completion of a given thread.
    * You have to deliver a reference to the thread you want 
    * your thread to wait for. The return value of the finished thread 
    * will be stored in the integer value pointed to by the second 
    * argument. If you do not need the return value you do not need to 
    * specify a pointer because the parameter defaults to zero.
    */
  virtual int join(const Thread&, int* rv=0);
  /** @ref join waits for completion of a given thread.
    * It differs from the previous function only in the type of the 
    * first argument. It is public because it 
    * may be useful if you want to wait for a thread which has not been 
    * started using the Threads class but with plain C code. The 
    * second argument also defaults to zero.
    */
  virtual int join(pthread_t, int* rv=0);
  /** A static method to join a thread. Use this one if you want to join a
   *  thread from outside of any Thread object (like in your main function).
   *  I am using a different name to avoid confusion.
   */
  static int joinThread(const Thread&, int *rv=0);
  /** @ref threadMain is the main function of the thread.
    * @ref threadMain is hopefully the most important function of a thread. 
    * On the other hand it is definitely simple in this base class: it 
    * is pure virtual. Every kind of thread MUST be implemented 
    * by deriving a class from @ref Thread and reimplementing the 
    * @ref threadMain (okay okay, it must, if you want to use this class:-).
    */
  virtual void* threadMain()=0;
  /** Returns the identity (pthread_t) of the started thread.
    * The identity of a thread is unique between all running threads 
    * inside ONE process, it is of the type @ref pthread_t which
    * is defined in pthread.h delivered with the C library 
    * (glibc) or with the linuxthreads package (libc5).
    * It is zero if no thread is running (that means you are able to 
    * determine whether there is a thread running or not by comparing 
    * identity() with zero).
    */
  pthread_t identity() const;
protected:
  /** A locked pthread_t storing the identity of this thread.
   */
  Lock<pthread_t> id;
  /** Since we need an address to a method this static function is
   *  used to create the threads.
   */
  static void* startRoutine(void*);
};


template <class type>
Lock<type>::Lock(pthread_mutexattr_t* attr)
{
  pthread_mutex_init(&mutex, attr);
}

template <class type>
Lock<type>::~Lock()
{
  int rc=pthread_mutex_destroy(&mutex);
  if(rc==EBUSY)
    debug("Lock::~Lock: Mutex about to be destroyed but not freed!");
  if(rc!=0)
    debug("Lock::~Lock: Error destroying the mutex.");
}

template <class type>
void Lock<type>::unlock() const
{
  int rc=pthread_mutex_unlock(&mutex);
  if(rc==EINVAL)
    debug("Lock::lock: Mutex not initialized "
	  "(constructor should do that).");
  // maybe later there are more error codes
  if(rc!=0)
    debug("Lock::lock: Error unlocking.");
}

template <class type>
type Lock<type>::get() const
{
  type temp;
  lock();
  temp=data;
  unlock();
  return temp;
}

template <class type>
void Lock<type>::set(const type& temp)
{
  lock();
  data=temp;
  unlock();
}

template <class type>
type* Lock<type>::lock() const
{
  int rc=pthread_mutex_lock(&mutex);
  if(rc==EINVAL)
    debug("Lock::lock: Mutex not initialized "
	  "(constructor should do that).");
  if(rc!=0)
    debug("Lock::lock: Error acquiring the lock.");
  if(rc==0) // everything OK
    {
      return &data;
    } else {
      return 0;
    }
}

template <class type>
type* Lock<type>::tryLock() const
{
  int rc=pthread_mutex_try_lock(&mutex);
  if(rc==EINVAL)
    debug("Lock::lock: Mutex not initialized "
	  "(constructor should do that).");
  if(rc!=0)
    debug("Lock::lock: Error acquiring the lock"
	  " (lock is busy).");
  if(rc==0) // everything OK
    {
      return &data;
    } else {
      return 0;
    }
}

#endif
